<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Request;
use Auth;
use Validator;
use Log;
use File;
use Storage;
use DB;
use Carbon\Carbon;
use Helper;
use App\Model\ProductBrands;
use Illuminate\Support\Str;

class ProductBrandsApiController extends Controller
{
    public function getProductBrandList(Request $request, $type = 1, $orderBy = 'order_by', $ascdesc = 'asc', $limit_page = 0, $keyword = '')
    {
        $query = ProductBrands::query();

        if ($keyword != '') {
            $query = $query->searchKeyword($keyword);
        }

        $query = $query->orderBy($orderBy, $ascdesc);
        if ($limit_page == 0) {
            $result = $query->get();
        } else {
            $result = $query->paginate($limit_page);
        }

        if ($type == 1) {
            return response()->json(array('status' => 'success', 'data' => $result), 200);
        } else {
            return $result;
        }
    }

    public function getProductBrandListFront(Request $request)
    {
        $query = ProductBrands::query();
        $query = $query->whereStatus("Active");
        $query = $query->orderBy("order_by", "ASC");
        $result = $query->get();
        return $result;
    }

    public function index(Request $request)
    {
        try {
            $query = ProductBrands::query();
            if ($request->input('where') !== null) {
                $where = $request->input('where');
                foreach ($where as $key => $value) {
                    $field = $key;
                    $cond = key($value);
                    $var = pos($value);
                    if ($cond == 7)
                        $var = "%" . $var . "%";
    
                    $query = $query->where($field, config("config.config_arr_condition")[$cond], $var);
                }
            }
    
            if ($request->input('order') !== null) {
                $order = $request->input('order');
                foreach ($order as $key => $value) {
                    $query = $query->orderBy($key, $value);
                }
            }
    
            if ($request->input('filter') !== null) {
                $filter = $request->input('filter');
    
                if ($filter["limit"] !== null && $filter["limit"] != 0) {
                    $query = $query->take($filter["limit"]);
                }
                if ($filter["offset"] !== null && $filter["offset"] != 0) {
                    $query = $query->offset(($filter["offset"] - 1));
                }
                if ($filter["skip"] !== null && $filter["skip"] != 0) {
                    $query = $query->skip($filter["skip"]);
                }
            }
    
            $data = $query->get();
    
            return response()->json([
                "success" => true,
                "total" => sizeof($data),
                'data' => $data,
                "msg" => "Get data success"
            ], 200);
        } catch (Exception $e) {
            return response()->json(array('success' => false, 'data' => $e, "message" => "Get all data fail"), 404);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title_th' => 'required',
            'title_en' => 'required',
            // 'url' => 'required|url',
        ],
        [
            'title_th.required' => 'กรุณาใส่ชื่อภาษาไทย',
            'title_en.required' => 'กรุณาใส่ชื่อภาษาอังกฤษ',
            // 'url.required' => 'กรุณาใส่ลิงค์',
            // 'url.url' => 'รูปแบบลิงค์ไม่ถูกต้อง! กรุณาใส่ลิงค์ใหม่',
        ]);

        if ($validator->fails()) {
            //return $validation->errors();
            return response()->json([
                "success" => false,
                'data' => null,
                "msg" => $validator->errors()
            ], 422);
        } else {
            $ip_address = $request->ip();
            if ($request->ip() == "::1") {
                $ip_address = "127.0.0.1";
            }

            $now = Carbon::now();
            $current_datetime = $now->toDateTimeString();

            $pathDate = $now->format("Y/m");
            $filenametostore = "";
            if ($request->hasFile('thumb')) {
                $pathFile = config('config.config_pathUpload') . '/productBrands/' . $pathDate;
                File::isDirectory($pathDate) or File::makeDirectory($pathFile, 0777, true, true);
                $filenametostore = 'thumb-' . time() . '.' . str_replace(" ", "_", $request->thumb->getClientOriginalName());
                Helper::storageProcess("save", $pathFile, $filenametostore, $request->file('thumb'));
            }

            $startDate = ($request->input("startDate") !== null) ? Carbon::createFromFormat('d/m/Y', $request->input("startDate"))->format("Y-m-d") : NULL;
            $endDate = ($request->input("endDate") !== null) ? Carbon::createFromFormat('d/m/Y', $request->input("endDate"))->format("Y-m-d") : NULL;

            $status = ($request->input("status") !== null) ? $request->input("status") : "Inactive";
            $order_by = 0;
            $created_by = ($request->input("created_by") !== null) ? $request->input("created_by") : 0;
            $updated_by = ($request->input("updated_by") !== null) ? $request->input("updated_by") : 0;

            $slug = Str::slug($request->input("title_en"));
            $countSlug = ProductBrands::whereSlug($slug)->count();
            if ($countSlug > 0)
                $slug = $slug.'-'.($countSlug+1);

            DB::beginTransaction();
            try {
                $productBrand = new ProductBrands;
                $productBrand->thumb = $filenametostore;
                $productBrand->title_th = $request->input("title_th");
                $productBrand->content_th = $request->input("content_th");
                $productBrand->title_en = $request->input("title_en");
                $productBrand->content_en = $request->input("content_en");
                $productBrand->slug = $slug;
                $productBrand->startDate = $startDate;
                $productBrand->endDate = $endDate;
                $productBrand->visitor = $ip_address;
                $productBrand->status = $status;
                $productBrand->created_at = $current_datetime;
                $productBrand->updated_at = $current_datetime;
                $productBrand->created_by = $created_by;
                $productBrand->updated_by = $updated_by;
                $productBrand->order_by = $order_by;
                $productBrand->save();

                $query = ProductBrands::query();
                $update_order = $query->increment("order_by");
                
                $data = array(
                    "productBrand" => $productBrand,
                );

                DB::commit();
                return response()->json(array('success' => true, 'data' => $data, "message" => "Successfully inserted"), 200);
            } catch (Exception $e) {
                DB::rollback();
                return response()->json(array('success' => false, 'data' => $e, "message" => "Inserted fail"), 404);
            }
        }
    }

    public function show($id)
    {
        $data = ProductBrands::find($id);
        if (empty($data)) {
            return response()->json([
                "success" => false,
                'data' => "",
                "msg" => ""
            ], 402);
        } else {
            return response()->json([
                "success" => true,
                "total" => $data->count(),
                'data' => $data,
                "msg" => "Get data by id success"
            ], 200);
        }
    }

    public function update(Request $request, $productBrand_id = 0)
    {
        $validator = Validator::make($request->all(), [
            'title_th' => 'required',
            'title_en' => 'required',
            // 'url' => 'required|url',
        ],
        [
            'title_th.required' => 'กรุณาใส่ชื่อภาษาไทย',
            'title_en.required' => 'กรุณาใส่ชื่อภาษาอังกฤษ',
            // 'url.required' => 'กรุณาใส่ลิงค์',
            // 'url.url' => 'รูปแบบลิงค์ไม่ถูกต้อง! กรุณาใส่ลิงค์ใหม่',
        ]);

        if ($validator->fails()) {
            //return $validation->errors();
            return response()->json([
                "success" => false,
                'data' => null,
                "msg" => $validator->errors()
            ], 422);
        } else {

            $ip_address = $request->ip();
            if ($request->ip() == "::1") {
                $ip_address = "127.0.0.1";
            }

            $productBrand = ProductBrands::find($productBrand_id);
            if (empty($productBrand)) {
                return response()->json(array('success' => false, 'data' => null, 'message' => 'No Data'), 200);
            }

            $now = Carbon::now();
            $current_datetime = $now->toDateTimeString();

            $pathDate = Carbon::parse($productBrand->created_at)->format("Y/m");
            $pathFile = config('config.config_pathUpload') . '/productBrands/' . $pathDate;

            $filenametostore = $productBrand->thumb;
            if ($request->action_thumb=="delete")
            {
                $responDelete = Helper::storageProcess("delete", $pathFile, $filenametostore);
                $filenametostore = "";
            }
            if ($request->hasFile('thumb')) {
                File::isDirectory($pathDate) or File::makeDirectory($pathFile, 0777, true, true);
                $new_filenametostore = 'thumb-' . time() . '.' . str_replace(" ", "_", $request->thumb->getClientOriginalName());

                $responUpload = Helper::storageProcess("save", $pathFile, $new_filenametostore, $request->file('thumb'));
                if ($responUpload)
                {
                    $responDelete = Helper::storageProcess("delete", $pathFile, $filenametostore);
                    $filenametostore = $new_filenametostore;
                }
            }

            $startDate = ($request->input("startDate") !== null) ? Carbon::createFromFormat('d/m/Y', $request->input("startDate"))->format("Y-m-d") : $productBrand->startDate;
            $endDate = ($request->input("endDate") !== null) ? Carbon::createFromFormat('d/m/Y', $request->input("endDate"))->format("Y-m-d") : $productBrand->endDate;

            $status = ($request->input("status") !== null) ? $request->input("status") : $productBrand->status;
            $order_by = ($request->input("order_by") !== null) ? $request->input("order_by") : $productBrand->order_by;
            $created_by = ($request->input("created_by") !== null) ? $request->input("created_by") : 0;
            $updated_by = ($request->input("updated_by") !== null) ? $request->input("updated_by") : 0;

            DB::beginTransaction();
            try {
                $productBrand->thumb = $filenametostore;
                $productBrand->title_th = $request->input("title_th");
                $productBrand->content_th = $request->input("content_th");
                $productBrand->title_en = $request->input("title_en");
                $productBrand->content_en = $request->input("content_en");
                $productBrand->startDate = $startDate;
                $productBrand->endDate = $endDate;
                $productBrand->visitor = $ip_address;
                $productBrand->status = $status;
                $productBrand->updated_at = $current_datetime;
                $productBrand->updated_by = $updated_by;
                $productBrand->order_by = $order_by;

                $productBrand->save();

                $data = array(
                    "productBrand" => $productBrand,
                );

                DB::commit();
                return response()->json(array('success' => true, 'data' => $data, "message" => "Successfully updated"), 200);
            } catch (Exception $e) {
                DB::rollback();
                return response()->json(array('success' => false, 'data' => $e, "message" => "Updated fail"), 404);
            }
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            $productBrand = ProductBrands::find($id);
            $productBrand->delete();

            $productBrands = ProductBrands::orderBy("order_by","asc")->get();
            if ($productBrands && $productBrands->count() > 0) {
                foreach ($productBrands as $key => $value) {
                    $dataUpdate = array(
                        "order_by" => $key + 1,
                    );
                    $value->update($dataUpdate);
                }
            }

            $data = array(
                "id" => $id
            );

            DB::commit();
            return response()->json([
                "success" => true,
                'data' => $data,
                "msg" => "ProductBrand - delete data success"
            ], 200);
        } catch (\Exception $e) {
            DB::rollback();
            return response()->json([
                "success" => false,
                'data' => "",
                "msg" => "ProductBrand - error"
            ], 502);
        }
    }
}